/*
 *  lighttemp.h
 *  
 *
 *  Created by Kat Dornian (Phidgets Inc) on 2014-05-31
 *  This file is part of nightlight.
 *  
 *  Nightlight is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  Nightlight is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You can see the GNU General Public License at 
 *  <http://www.gnu.org/licenses/>.
 *
 */

#include <stdio.h>
#include <phidget21.h>
#include <unistd.h>

#include "lighttemp.h"

int CCONV AttachHandler(CPhidgetHandle IFK, void *userptr)
{
	int serialNo;
	const char *name;
	
	CPhidget_getDeviceName(IFK, &name);
	CPhidget_getSerialNumber(IFK, &serialNo);
	
	printf("%s %10d attached!\n", name, serialNo);
	
	return 0;
}

int CCONV DetachHandler(CPhidgetHandle IFK, void *userptr)
{
	int serialNo;
	const char *name;
	
	CPhidget_getDeviceName (IFK, &name);
	CPhidget_getSerialNumber(IFK, &serialNo);
	
	printf("%s %10d detached!\n", name, serialNo);
	
	return 0;
}

int CCONV ErrorHandler(CPhidgetHandle IFK, void *userptr, int ErrorCode, const char *unknown)
{
	printf("Error handled. %d - %s", ErrorCode, unknown);
	return 0;
}

// Callback that will run if the digital input changes.
// Index - Index of the input that generated the event
// State - boolean (0 or 1) representing the input state (on or off)
int CCONV InputChangeHandler(CPhidgetInterfaceKitHandle IFK, void *usrptr, int Index, int State)
{	
	int currentState = PFALSE;
	int i, returnTemp;
	double actualTemp;
	
	// Respond to the rug switch first being triggered, i.e. changing to true
	if (Index == RUG_SWITCH_INPUT && State == PTRUE) {
		
		// Check the if the light is already on, if it is, turn the light off
		// Note: we assume LEDs are attached to digital outputs 0 through 6
		for (i=0; i < 6 && currentState == PFALSE; i++) {
			CPhidgetInterfaceKit_getOutputState(IFK, i, &currentState);
		}
		
		if(currentState == PTRUE) {
			for (i=0; i < 6; i++) {
				CPhidgetInterfaceKit_setOutputState(IFK, i, PFALSE);
			}
		} // end if(currentState == PTRUE)
	
		
		else { // The light is off...
			
			// Time to turn on the appropriate LEDs based on temperature
			CPhidgetInterfaceKit_getSensorValue (IFK, TEMPERATURE, &returnTemp);
			actualTemp = (returnTemp*0.22222) - 61.11;
			fprintf(stdout, "\nThe current temperature is %lf\n", actualTemp);
			
			if (actualTemp <= LOWEST_TEMP) { // deep blue
				CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PTRUE);
			}
			else if (actualTemp <= TEMP_1) { // icy green
				CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PTRUE);
			}
			else if (actualTemp <= TEMP_2) { // purple
				CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PTRUE); 
			}
			else if (actualTemp <= TEMP_3) { // pure green
				CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PFALSE); 
			}
			else if (actualTemp <= HIGHEST_TEMP) { // natural yellow 
				CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PFALSE); 
			}
			else if (actualTemp > HIGHEST_TEMP) { // hot red 
				CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PFALSE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PFALSE); 
			}
			else { // white
				CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PTRUE); 
				CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PTRUE); 
			}
			
		} // end else, that is if(currentState != PTRUE)

	} // end if (Index == RUG_SWITCH_INPUT && State == PTRUE)
	return 0;
}

// Callback that will run if an output changes.
// Index - Index of the output that generated the event
// State - boolean (0 or 1) representing the output state (on or off)
int CCONV OutputChangeHandler(CPhidgetInterfaceKitHandle IFK, void *usrptr, int Index, int State)
{
	// printf("Digital Output: %d > State: %d\n", Index, State);
	return 0;
}

// Callback that will run if the sensor value changes by more than the OnSensorChange trigger.
// Index - Index of the sensor that generated the event
// Value - the sensor read value
int CCONV SensorChangeHandler(CPhidgetInterfaceKitHandle IFK, void *usrptr, int Index, int Value)
{
	int i, returnTemp, currentState = PFALSE;
	double actualTemp;
	
	// Check the if the light is already on, if it is, turn the light off
	// Note: we assume LEDs are attached to digital outputs 0 through 6
	for (i=0; i < 6 && currentState == PFALSE; i++) {
		CPhidgetInterfaceKit_getOutputState(IFK, i, &currentState);
	}
	
	// Coming near the sensor will switch the state of the lights
	// Note: we assume LEDs are attached to digital outputs 0 through 6
	if ( Index == DISTANCE && Value < 500 ) {
		
		for (i=0; i < 6; i++) {
			CPhidgetInterfaceKit_setOutputState(IFK, i, !currentState);
		}
		
	} // end if ( Index == DISTANCE && Value < 500 )
	
	// If the temperature changed and the lights are on, show the new temperature
	else if ( Index == TEMPERATURE && currentState == PTRUE) {
	
		// Check if it's appropriate to change the temperature
		CPhidgetInterfaceKit_getSensorValue (IFK, TEMPERATURE, &returnTemp);
		actualTemp = (returnTemp*0.22222) - 61.11;
		fprintf(stdout, "\nThe current temperature is %lf\n", actualTemp);
		
		if (actualTemp <= LOWEST_TEMP) { // deep blue
			CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PTRUE);
		}
		else if (actualTemp <= TEMP_1) { // icy green
			CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PTRUE);
		}
		else if (actualTemp <= TEMP_2) { // purple
			CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PTRUE); 
		}
		else if (actualTemp <= TEMP_3) { // pure green
			CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PFALSE); 
		}
		else if (actualTemp <= HIGHEST_TEMP) { // natural yellow 
			CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PFALSE); 
		}
		else if (actualTemp > HIGHEST_TEMP) { // hot red 
			CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PFALSE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PFALSE); 
		}
		else { // white
			CPhidgetInterfaceKit_setOutputState(IFK, RED_1, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, RED_2, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_1, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, GREEN_2, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_1, PTRUE); 
			CPhidgetInterfaceKit_setOutputState(IFK, BLUE_2, PTRUE); 
		}
		
	} // end else if ( Index == TEMPERATURE && currentState == PTRUE)
	
	return 0;
}

int interfacekit_simple()
{
	int result, numSensors, i;
	const char *err;
	
	//Declare an InterfaceKit handle
	CPhidgetInterfaceKitHandle ifKit = 0;
	
	//create the InterfaceKit object
	CPhidgetInterfaceKit_create(&ifKit);
	
	//Set the handlers to be run when the device is plugged in or opened from software, unplugged or closed from software, or generates an error.
	CPhidget_set_OnAttach_Handler((CPhidgetHandle)ifKit, AttachHandler, NULL);
	CPhidget_set_OnDetach_Handler((CPhidgetHandle)ifKit, DetachHandler, NULL);
	CPhidget_set_OnError_Handler((CPhidgetHandle)ifKit, ErrorHandler, NULL);
	
	//Registers a callback that will run if an input changes.
	//Requires the handle for the Phidget, the function that will be called, and an arbitrary pointer that will be supplied to the callback function (may be NULL).
	CPhidgetInterfaceKit_set_OnInputChange_Handler (ifKit, InputChangeHandler, NULL);
	
	//Registers a callback that will run if the sensor value changes by more than the OnSensorChange trig-ger.
	//Requires the handle for the IntefaceKit, the function that will be called, and an arbitrary pointer that will be supplied to the callback function (may be NULL).
	CPhidgetInterfaceKit_set_OnSensorChange_Handler (ifKit, SensorChangeHandler, NULL);
	
	//Registers a callback that will run if an output changes - we won't care about when the output changes... just noise
	//Requires the handle for the Phidget, the function that will be called, and an arbitrary pointer that will be supplied to the callback function (may be NULL).
	//CPhidgetInterfaceKit_set_OnOutputChange_Handler (ifKit, OutputChangeHandler, NULL);
	
	//open the interfacekit for device connections
	CPhidget_open((CPhidgetHandle)ifKit, -1);
	
	//get the program to wait for an interface kit device to be attached
	printf("Waiting for interface kit to be attached....");
	if((result = CPhidget_waitForAttachment((CPhidgetHandle)ifKit, 10000)))
	{
		CPhidget_getErrorDescription(result, &err);
		printf("Problem waiting for attachment: %s\n", err);
		return 0;
	}
	
	// Settings for sensors: 
	CPhidgetInterfaceKit_setSensorChangeTrigger (ifKit, TEMPERATURE, 9); // this is about 2 degrees Celsius
	CPhidgetInterfaceKit_setSensorChangeTrigger (ifKit, DISTANCE, 450);
	
	// Run the program forever
	while (1) {
		usleep(5000);
	}

	for (i=0; i < 6; i++) {
		CPhidgetInterfaceKit_setOutputState(ifKit, i, PFALSE);
	}
	
	CPhidget_close((CPhidgetHandle)ifKit);
	CPhidget_delete((CPhidgetHandle)ifKit);
	
	//all done, exit
	return 0;
}

int main(int argc, char* argv[])
{	
	interfacekit_simple();
	return 0;
}




